<?php

namespace App\Http\Controllers\Admin;

use App\Models\Blog;
use Inertia\Inertia;
use App\Models\Brand;
use App\Utils\Helper;
use App\Enums\BlogType;
use App\Enums\UserType;
use App\Enums\ImageType;
use App\Models\Category;
use App\Models\BlogImage;
use App\Rules\UniqueSlug;
use App\Enums\LanguageType;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use Illuminate\Validation\Rule;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Http\Services\SoftDeleteService;

class BlogController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $blog = Blog::with('category')
      ->filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "blog" => collect($blog),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Blog/Index", $data);
  }

  protected function getFormData($request, $blog = null)
  {
    $allCategories = BlogCategory::select(["id", "lang", "id as value", "name as label"])->where("status", PublishStatusType::Publish)->get();

    $categories = [];
    $validation = $this->validation($request);
    foreach ($validation['langs'] as $lang) {
      $categories[$lang] = $allCategories->where('lang', $lang)->values();
    }

    $data = [
      "categories" => $categories,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Blog/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $blog = Blog::with('translatedGroup.blog', 'blogImage')->where("id", $id)->first();
    if (!$blog) return Helper::redirectBack('error','Data tidak ditemukan');  

    if (!$this->getAuthorize(request(), $blog))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($blog->translatedGroup->blog as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $blog->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $blog = collect($blog)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $blog['images'] = $blog['blog_image'];

    $formData = $this->getFormData($request, $blog);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "blog" => $blog,
    ];

    return Inertia::render("Admin/Blog/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        unset($data['images']);
        $data['slug'] = Helper::slugify($data['slug']);
        $blog = Blog::create($data);
        $this->saveBlogImage($request, $blog);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    if ($request->restore_data == '1') return SoftDeleteService::restore("categories", $id, $this->page);
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $blog = Blog::where("id", $id)->first();
      if (!$blog) return Helper::redirectBack('error','Data tidak ditemukan');  

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $blog);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        unset($data['images']);
        if (empty($data['id'])) {
          continue;
        }
        $data['slug'] = Helper::slugify($data['slug']);

        $blog = Blog::firstWhere('id', $data['id']);
        $blog->update($data);
        $this->saveBlogImage($request, $blog);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $blog = null)
  {
    if ($request->file('thumbnail')) {
      $directory = 'blog';
      if (!Storage::disk('public')->exists($directory)) {
        Storage::disk('public')->makeDirectory($directory);
      }

      $fileExt = $request->file('thumbnail')->getClientOriginalExtension();
      $fileName = config('app.name') . "-" . strtolower($request->slug_id) ."-". time() . ".$fileExt";
      $file = Storage::disk('public')->putFileAs($directory, $request->file('thumbnail'), $fileName);
      if (!$file) redirect()->back()->with('error', 'Gagal upload file');
      $validatedData['thumbnail'] = "/storage/$file";
    }

    $filteredData = [];
    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Blog
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function saveBlogImage($request, $blog)
  {
    $images = $request->file('images');

    $latestImage = collect($request->images)->pluck('id')->whereNotNull()->toArray();
    $blogImage = BlogImage::where('blog_id', $blog->id)->whereNotIn('id', $latestImage)->get();

    if ($blogImage->isNotEmpty()) {
      foreach ($blogImage as $image) {
        if (Storage::disk('public')->exists(substr($image->value, 9))) {
          Storage::disk('public')->delete(substr($image->value, 9));
        }
        $image->delete();
      }
    }

    if (!$images) return;

    $directory = 'blog';
    if (!Storage::disk('public')->exists($directory)) {
      Storage::disk('public')->makeDirectory($directory);
    }

    if (!empty($images)) {
      foreach ($images as $image) {
        $fileName = config('app.name') . "-" . str($image->getClientOriginalName())->slug()->value()  . "-" . time(). "-" . rand(111,999) . "." . $image->getClientOriginalExtension();
        $file = Storage::disk('public')->putFileAs($directory, $image, $fileName);
        if (!$file) redirect()->back()->with('error', 'Gagal upload file');

        if (!isset($image->id)) {
          BlogImage::create([
            'user_id' => auth()->id(),
            'blog_id' => $blog->id,
            'type' => ImageType::File,
            'label' => $image->getClientOriginalName(),
            'value' => "/storage/$file"
          ]);
        }
      }
    }
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $blog = Blog::with('translatedGroup.blog')->whereId($id)->first();
      if (!$blog) return Helper::redirectBack('error','Data tidak ditemukan');  

      if (!$this->getAuthorize(request(), $blog))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      foreach ($blog->translatedGroup->blog as $translated) {
        $translated->delete();
      }
      $blog->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $blog = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "blog",
      "label" => "Artikel",
      "url" => "/admin/blog/article",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $table = (new Blog)->getTable();

    $data = [
      "validation" => [
        'category_id'  => "required",
        'id'  => "nullable",
        'lang'  => "required",
        'title'  => "required",
        "slug" => "required|unique:$table,slug",
        'description'  => "required",
        'keyword'  => "required",
        'short_description'  => "required",
        'status'  => "required",
        'thumbnail'  => "required",
        'images'  => "required_if:is_slider,1",
        'is_slider'  => "nullable",
        'published_at'  => "required_if:status," . PublishStatusType::Publish,
      ],
      "default" => [
        "lang" => LanguageType::Id,
        'is_slider'  => 0,
        "status" => PublishStatusType::Draft,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "category_id" => [],
        "title" => [],
        "slug" => [],
        "description" => [],
        "keyword" => [],
        "short_description" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
              $data['validation']["{$column}_{$lang}"] = ['required', new UniqueSlug($table, $lang, [$request->input("id_$lang")])];
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
