<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Models\Brand;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class BrandController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $brand = Brand::filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "brand" => collect($brand),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Brand/Index", $data);
  }

  protected function getFormData($request, $brand = null)
  {
    $data = [
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Brand/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $brand = Brand::where("id", $id)->first();
    if (!$brand) return Helper::redirectBack('error','Data tidak ditemukan');  

    if (!$this->getAuthorize(request(), $brand))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $formData = $this->getFormData($request, $brand);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "brand" => $brand,
    ];

    return Inertia::render("Admin/Brand/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      $validatedData['slug'] = Helper::slugify($validatedData['slug']);
      $existSlug = Brand::where(['slug' => $validatedData['slug']])->first();
      $brand = Brand::create($validatedData);

      if (!empty($existSlug)) {
        $brand->slug = "{$brand->id}-{$brand->slug}";
        $brand->save();
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $brand = Brand::where("id", $id)->first();
      if (!$brand) return Helper::redirectBack('error','Data tidak ditemukan');  

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $brand);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      $validatedData['slug'] = Helper::slugify($validatedData['slug']);
      $existSlug = Brand::where(['slug' => $validatedData['slug']])->first();
      $brand->update($validatedData);

      if (!empty($existSlug)) {
        $brand->slug = "{$brand->id}-{$brand->slug}";
        $brand->save();
      }
      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $brand = null)
  {
      if ($request->file('image')) {
      $directory = 'brand';
      if (!Storage::disk('public')->exists($directory)) {
        Storage::disk('public')->makeDirectory($directory);
      }

      $fileExt = $request->file('image')->getClientOriginalExtension();
      $fileName = config('app.name') . "-" . strtolower($request->slug) . time() . ".$fileExt";
      $file = Storage::disk('public')->putFileAs($directory, $request->file('image'), $fileName);
      if (!$file) redirect()->back()->with('error', 'Gagal upload file');
      $validatedData['image'] = "/storage/$file";
    }

    if(empty($validatedData['slug'])){
      $validatedData['slug'] = str($validatedData['name'])->slug()->value;
    }

    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $brand = Brand::whereId($id) ->first();
      if (!$brand) return Helper::redirectBack('error','Data tidak ditemukan');  

      if (!$this->getAuthorize(request(), $brand))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      $brand->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $brand = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "brand",
      "label" => "Merek",
      "url" => "/admin/brand",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $data = [
      "validation" => [
        'id'  => "nullable",
        'name'  => "required",
        'slug'  => "nullable",
        'image'  => "nullable",
        'status'  => "required",
      ],
      "default" => [
        "status" => PublishStatusType::Draft,
      ],
      "langs" => [],
      "lang_columns" => []
    ];

    return $data;
  }
}
