<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Models\Gallery;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class GalleryController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $gallery = Gallery::filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "gallery" => collect($gallery),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Gallery/Index", $data);
  }

  protected function getFormData($request, $gallery = null)
  {
    $data = [
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Gallery/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $gallery = Gallery::with('translatedGroup.gallery')->where("id", $id)->first();
    if (!$gallery) return Helper::redirectBack('error','Data tidak ditemukan');  

    if (!$this->getAuthorize(request(), $gallery))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($gallery->translatedGroup->gallery as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $gallery->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $gallery = collect($gallery)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $formData = $this->getFormData($request, $gallery);

    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "gallery" => $gallery,
    ];

    return Inertia::render("Admin/Gallery/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        Gallery::create($data);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $gallery = Gallery::where("id", $id)->first();
      if (!$gallery) return Helper::redirectBack('error','Data tidak ditemukan');  

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $gallery);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        if (empty($data['id'])) {
          continue;
        }
        $gallery = Gallery::firstWhere('id', $data['id']);
        $gallery->update($data);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $gallery = null)
  {
    if ($request->file('image')) {
      $directory = 'gallery';
      if (!Storage::disk('public')->exists($directory)) {
        Storage::disk('public')->makeDirectory($directory);
      }

      $fileExt = $request->file('image')->getClientOriginalExtension();
      $fileName = str($request->description_id)->slug()->limit(50, '')->value();
      $fileName .= "-" . time();
      $fileName = config('app.name') . "-" . $fileName . ".$fileExt";
      $file = Storage::disk('public')->putFileAs($directory, $request->file('image'), $fileName);
      if (!$file) redirect()->back()->with('error', 'Gagal upload file');
      $validatedData['image'] = "/storage/$file";
    }

    $filteredData = [];
    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Gallery
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $gallery = Gallery::with('translatedGroup.gallery')->whereId($id) ->first();
      if (!$gallery) return Helper::redirectBack('error','Data tidak ditemukan');  

      if (!$this->getAuthorize(request(), $gallery))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      foreach ($gallery->translatedGroup->gallery as $translated) {
        $translated->delete();
      }

      $gallery->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $gallery = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "gallery",
      "label" => "Galeri Proyek",
      "url" => "/admin/gallery",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $data = [
      "validation" => [
        'id'  => "nullable",
        'lang'  => "required",
        'image'  => "required",
        'description'  => "required",
        'status'  => "required",
      ],
      "default" => [
        "lang" => LanguageType::Id,
        "status" => PublishStatusType::Draft,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "description" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
            $data['validation']["{$column}_{$lang}"] .= "," . $request->input("id_$lang");
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
