<?php

namespace App\Http\Controllers\Admin;

use App\Enums\ProfileActionType;
use App\Enums\ProfileFormType;
use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Models\Profile;
use App\Enums\ProfileType;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use Illuminate\Validation\Rule;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Http\Services\SoftDeleteService;

class ProfileController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;
  public $formAction = null;
  public $createableForm = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);

      $this->formAction = collect(ProfileActionType::getInstances())
        ->filter(fn($action) => is_array($action->value) && in_array('create', $action->value))
        ->keys()
        ->all();

      $this->createableForm = ProfileType::getValues($this->formAction);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $profile = Profile::filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->when($request->filled("type"), function ($query) use ($request) {
        $query->where("type", $request->type);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $imageTypes = [
      ProfileType::AboutImage
    ];

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "imageTypes" => $imageTypes,
      "status" => PublishStatusType::getValues(),
      "types" => ProfileType::getValues(),
      "page" => $this->page,
      "profile" => collect($profile),
      "isAdmin" => $this->isAdmin,
      "formAction" => $this->formAction,
      "createableForm" => $this->createableForm,
    ];

    return Inertia::render("Admin/Profile/Index", $data);
  }

  protected function getFormData($request, $profile = null)
  {
    $formTypes = [];
    foreach (ProfileType::getKeys() as $key) {
      $type = ProfileType::getValue($key);
      $formTypes[$type] = ProfileFormType::getValue($key);
    }

    $data = [
      "status" => PublishStatusType::getValues(),
      "types" => ProfileType::getValues(),
      "formTypes" => $formTypes,
      "formAction" => $this->formAction,
      "createableForm" => $this->createableForm,
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $formData['types'] = ProfileType::getValues($formData['formAction']);

    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Profile/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $profile = Profile::with('translatedGroup.profile')->where("id", $id)->first();
    if (!$profile) return Helper::redirectBack('error', 'Data tidak ditemukan');

    if (!$this->getAuthorize(request(), $profile))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($profile->translatedGroup->profile as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $profile->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $profile = collect($profile)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $formData = $this->getFormData($request, $profile);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "profile" => $profile,
    ];

    return Inertia::render("Admin/Profile/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $actionTypes = ProfileActionType::getValue(ProfileType::getKey($validatedData['type']));
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        $profile = null;
        if (in_array('update', $actionTypes)) {
          $profile = Profile::where(['type' => $data['type'], 'lang' => $data['lang']])->first();
        }

        if (!empty($profile) && !in_array('create', $actionTypes)) {
          $profile->update($data);
          continue;
        }

        $profile = Profile::create($data);
        Helper::savePageCache($this->page, 'type', $profile);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $profile = Profile::where("id", $id)->first();
      if (!$profile) return Helper::redirectBack('error', 'Data tidak ditemukan');

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $profile);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        if (empty($data['id'])) {
          continue;
        }

        $profile = Profile::firstWhere('id', $data['id']);
        $profile->update($data);

        Helper::savePageCache($this->page, 'type', $profile);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $profile = null)
  {
    $filteredData = [];
    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Profile
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
          if ($request->file("content_$lang") && $column == 'content') {
            $directory = 'about';
            if (!Storage::disk('public')->exists($directory)) {
              Storage::disk('public')->makeDirectory($directory);
            }

            $fileExt = $request->file("content_$lang")->getClientOriginalExtension();
            $fileName = $request->file("content_$lang")->getClientOriginalName();
            $fileName = str($fileName)->slug()->limit(50, '')->value();
            $fileName = config('app.name') . "-" . $fileName . ".$fileExt";
            $file = Storage::disk('public')->putFileAs($directory, $request->file("content_$lang"), $fileName);
            if (!$file) redirect()->back()->with('error', 'Gagal upload file');
            $validatedColumns[$column] = "/storage/$file";
          }
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $profile = Profile::with('translatedGroup.profile')->whereId($id)->first();
      if (!$profile) return Helper::redirectBack('error', 'Data tidak ditemukan');

      if (!$this->getAuthorize(request(), $profile))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      if (!in_array($profile->type, $this->createableForm))
        return redirect()->back()->with("error", "Profil ini hanya dapat di Edit");

      foreach ($profile->translatedGroup->profile as $translated) {
        $translated->delete();
      }

      $profile->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $profile = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "profile",
      "label" => "Profil",
      "url" => "/admin/profile",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $table = (new Profile)->getTable();

    $data = [
      "validation" => [
        "label" => [Rule::requiredIf(function () use ($request) {
          return in_array($request->type, [ProfileType::MarketSegmentList]);
        })],
        'id'  => "nullable",
        'lang'  => "required",
        'type'  => "required",
        'content'  => "required",
        'status'  => "required",
      ],
      "default" => [
        "type" => ProfileType::CompanyName,
        "lang" => LanguageType::Id,
        "status" => PublishStatusType::Publish,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "label" => [],
        "content" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
            $data['validation']["{$column}_{$lang}"] .= "," . $request->input("id_$lang");
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
