<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Models\Seo;
use App\Enums\SeoType;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class SeoController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $seo = Seo::filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->when($request->filled("type"), function ($query) use ($request) {
        $query->where("type", $request->type);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "status" => PublishStatusType::getValues(),
      "types" => SeoType::getValues(),
      "page" => $this->page,
      "seo" => collect($seo),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Seo/Index", $data);
  }

  protected function getFormData($request, $seo = null)
  {
    $data = [
      "status" => PublishStatusType::getValues(),
      "types" => SeoType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function edit($id, Request $request)
  {
    $seo = Seo::with('translatedGroup.seo')->where("id", $id)->first();
    if (!$seo) return Helper::redirectBack('error','Data tidak ditemukan');  

    if (!$this->getAuthorize(request(), $seo))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($seo->translatedGroup->seo as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $seo->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $seo = collect($seo)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $formData = $this->getFormData($request, $seo);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "seo" => $seo,
    ];

    return Inertia::render("Admin/Seo/Edit", $data);
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $seo = Seo::where("id", $id)->first();
      if (!$seo) return Helper::redirectBack('error','Data tidak ditemukan');  

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $seo);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        if (empty($data['id'])) {
          continue;
        }

        $seo = Seo::firstWhere('id', $data['id']);
        $seo->update($data);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $seo = null)
  {
    $filteredData = [];
    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Seo
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $seo = Seo::with('translatedGroup.seo')->whereId($id) ->first();
      if (!$seo) return Helper::redirectBack('error','Data tidak ditemukan');  

      if (!$this->getAuthorize(request(), $seo))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      foreach ($seo->translatedGroup->seo as $translated) {
        $translated->delete();
      }

      $seo->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $seo = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "seo",
      "label" => "SEO",
      "url" => "/admin/seo",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $table = (new Seo)->getTable();

    $data = [
      "validation" => [
        'id'  => "nullable",
        'lang'  => "required",
        'type'  => "required",
        'title'  => "required",
        'seo_keyword'  => "required",
        'seo_description'  => "required",
        'status'  => "required",
      ],
      "default" => [
        "type" => SeoType::Default,
        "lang" => LanguageType::Id,
        "status" => PublishStatusType::Publish,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "title" => [],
        "seo_keyword" => [],
        "seo_description" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
            $data['validation']["{$column}_{$lang}"] .= "," . $request->input("id_$lang");
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
