<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Enums\LanguageType;
use App\Models\Translation;
use Illuminate\Http\Request;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class TranslationController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $langs = LanguageType::getValues();
    $translationQuery = Translation::get();

    $translations = [];
    foreach ($langs as $lang) {
      $translation = $translationQuery->where('lang', $lang)->first();
      if ($translation) {
        $translations[] = $translation->toArray();
        continue;
      }

      $translations[] = [
        'lang' => $lang,
        'content' => [],
      ];
    }

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => $langs,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "translations" => collect($translations),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Translation/Index", $data);
  }

  protected function getFormData($request, $translation = null)
  {
    $data = [
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $translation = null;
      if ($id != 'store') {
        $translation = Translation::where("id", $id)->first();
        if (!$translation) return Helper::redirectBack('error', 'Data tidak ditemukan');
      }

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $translation);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      if ($translation) {
        $translation->update($validatedData);
      } else {
        Translation::create($validatedData);
      }
      
      DB::commit();
      
      if ($id == 'store') {
        return Inertia::location($this->page['url']);
      }

      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $translation = null)
  {
    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $translation = Translation::whereId($id) ->first();
      if (!$translation) return Helper::redirectBack('error', 'Data tidak ditemukan');

      if (!$this->getAuthorize(request(), $translation))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      $translation->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $translation = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "translation",
      "label" => "Terjemahan",
      "url" => "/admin/translation",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $data = [
      "validation" => [
        'id'  => "nullable",
        'lang'  => "nullable",
        'content'  => "nullable",
      ],
      "default" => [],
      "langs" => [],
      "lang_columns" => []
    ];

    return $data;
  }
}
