<?php

namespace App\Http\Controllers\User;

use App\Models\Chat;
use App\Models\Order;
use App\Utils\Helper;
use App\Enums\ChatType;
use App\Enums\UserType;
use App\Enums\ImageType;
use App\Models\ChatMessage;
use Illuminate\Http\Request;
use App\Enums\ChatStatusType;
use App\Models\ChatAttachment;
use Illuminate\Cache\RateLimiter;
use Illuminate\Support\Facades\DB;
use App\Enums\ChatMessageStatusType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class ChatController extends Controller
{
  public function store(Request $request, RateLimiter $limiter)
  {
    $validatedData = $request->validate([
      ...$this->chatMessageValidation($request)["validation"],
      "attachment.*" => "file|mimes:jpg,jpeg,png,webp,avif,pdf|max:5120"
    ]);

    DB::beginTransaction();
    try {
      $chat = null;
      if ($request->filled('chat_id') && $request->chat_id != 'new') {
        $chat = Chat::where('id', $request->chat_id)->whereNot('status', ChatStatusType::Close)->first();
      }

      $user = auth()->check() ? auth()->user() : null;

      if (!$chat) {
        $now = microtime(true);
        $microtimeFloat = $now * 1000000;
        $microtimeString = number_format($microtimeFloat, 0, '', '');
        $visitorNameId = 'V' . $microtimeString;

        $user_agent = $request->userAgent();
        $ip_address = $request->ip();
        $deviceData = [
          'user_agent' => $user_agent,
          'ip_address' => $ip_address,
        ];

        $deviceId = hash('sha256', json_encode($deviceData));

        $chatData = [
          'user_id' => $user?->id ?? null,
          'name' => $visitorNameId,
          'device_id' => $deviceId,
          'user_agent' => $user_agent,
          'ip_address' => $ip_address,
        ];

        $chat = Chat::create($chatData);
      }

      $validatedData['chat_id'] = $chat->id;
      $validatedData['user_id'] = $user?->id ?? null;
      $validatedData['is_admin'] = $user?->role == UserType::Admin;

      if (! $validatedData['is_admin']) {
        if ($user) {
          $key = 'chat:user:' . $user->id;
          $maxAttempts = 30;
          $decaySeconds = 60;
        } else {
          $devicePart = $chat->device_id ?: hash('sha256', $request->ip() ?? '');
          $key = 'chat:visitor:' . $devicePart;
          $maxAttempts = 5;
          $decaySeconds = 60;
        }

        if ($limiter->tooManyAttempts($key, $maxAttempts)) {
          $seconds = $limiter->availableIn($key);
          DB::rollBack();

          return redirect()->back()
            ->with('error', 'You have sent too many messages. Please wait a moment and try again.')
            ->with('retry_after_seconds', $seconds);
        }

        $limiter->hit($key, $decaySeconds);
      }

      DB::commit();
      return $this->chat($validatedData, $chat, $user, $request);
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Pesan Gagal Dikirim. Cobalah beberapa saat lagi");
    }
  }
  public function status($id, Request $request)
  {
    $userId = auth()->check() ? auth()->user()->id : null;
    $chat = Chat::with('message')->withCount(['message as new_message_count' => function ($query) use ($userId) {
      $query->where('status', ChatMessageStatusType::Sent)->where('user_id', '!=', $userId);
    }])->where('id', $id)->first();

    $data = [
      'status' => 200,
      'message' => 'Berhasil mendapatkan status percakapan',
    ];

    if (!$chat) {
      $data = [
        'status' => 404,
        'message' => 'Data percakapan tidak ditemukan',
      ];
    }
    $is_new = $chat->new_message_count > 0;

    if ($is_new) {
      ChatMessage::where('chat_id', $chat->id)
        ->where('status', ChatMessageStatusType::Sent)
        ->where('user_id', '!=', $userId)
        ->update(['status' => ChatMessageStatusType::Read]);
    }

    $data['data'] = [
      'chat' => $chat,
      'is_new' => $is_new
    ];

    return response()->json($data, $data['status']);
  }

  public function chat($validatedData, $chat, $user, $request)
  {
    DB::beginTransaction();
    try {
      $successMessage = "";
      if (auth()->check() && $user?->role == UserType::Admin) {
        ChatMessage::where('chat_id', $chat->id)
          ->where('status', ChatMessageStatusType::Sent)
          ->update(['status' => ChatMessageStatusType::Read]);

        $chat->update(['status' => ChatStatusType::Open]);
        
        $successMessage = "Pesan Berhasil Dikirim";
      }

      $chatMessage = ChatMessage::create($validatedData);
      $minutes = now()->diffInMinutes(now()->addYear());
      $cookie = cookie('chat_id', base64_encode($chat->id), $minutes, '/', null, false, false);

      $this->saveChatAttachment($request, $chatMessage);

      if ($request->ref == 'backoffice' && in_array($user?->role, [UserType::Admin, UserType::Editor])) {
        $cookie = cookie('chat_id', '', $minutes);
      }

      DB::commit();
      return redirect()->back()->with("success", $successMessage)->withCookie($cookie);
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Pesan Gagal Dikirim. Cobalah beberapa saat lagi");
    }
  }

  public function saveChatAttachment($request, $chatMessage)
  {
    $attachments = $request->file('attachment');

    $latestImage = collect($request->attachments)->pluck('id')->whereNotNull()->toArray();
    $chatAttachment = ChatAttachment::where('chat_message_id', $chatMessage->id)->whereNotIn('id', $latestImage)->get();

    if ($chatAttachment->isNotEmpty()) {
      foreach ($chatAttachment as $image) {
        if (Storage::disk('public')->exists(substr($image->value, 9))) {
          Storage::disk('public')->delete(substr($image->value, 9));
        }
        $image->delete();
      }
    }

    if (!$attachments) return;

    $directory = 'chat_attachment';
    if (!Storage::disk('public')->exists($directory)) {
      Storage::disk('public')->makeDirectory($directory);
    }

    if (!empty($attachments)) {
      foreach ($attachments as $attachment) {
        $fileName = config('app.name') . "-" . str($attachment->getClientOriginalName())->slug()->value()  . "-" . time(). "-" . rand(111,999) . "." . $attachment->getClientOriginalExtension();
        $file = Storage::disk('public')->putFileAs($directory, $attachment, $fileName);
        if (!$file) redirect()->back()->with('error', 'Gagal upload file');

        if (!isset($attachment->id)) {
          ChatAttachment::create([
            'chat_id' => $chatMessage->chat_id,
            'chat_message_id' => $chatMessage->id,
            'type' => ImageType::File,
            'label' => $attachment->getClientOriginalName(),
            'is_image' => str_starts_with($attachment->getMimeType(), 'image/'),
            'value' => "/storage/$file"
          ]);
        }
      }
    }
  }

  protected function chatMessageValidation($request, $id = null)
  {
    return [
      "validation" => [
        "chat_id" => "nullable",
        "message" => "required|max:500",
        "status" => "nullable",
        "attachment" => "nullable|array",
      ],
      "default" => [
        "status" => ChatMessageStatusType::Sent,
      ]
    ];
  }
}
