<?php

namespace App\Http\Middleware;

use App\Models\Seo;
use App\Models\Chat;
use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\SeoType;
use App\Models\Profile;
use Inertia\Middleware;
use App\Enums\ProfileType;
use Tightenco\Ziggy\Ziggy;
use App\Models\Translation;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Configuration;
use App\Enums\OriginStatusType;
use App\Enums\ConfigurationType;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Session;

class HandleInertiaRequests extends Middleware
{
  /**
   * The root template that is loaded on the first page visit.
   *
   * @var string
   */
  protected $rootView = 'app';

  /**
   * Determine the current asset version.
   */
  public function version(Request $request): string|null
  {
    return parent::version($request);
  }

  /**
   * Define the props that are shared by default.
   *
   * @return array<string, mixed>
   */
  public function share(Request $request): array
  {
    if (str(env('APP_URL'))->startsWith('https://')) {
      Inertia::encryptHistory(true);
    }

    $lang = Helper::getLangActive($request);
    $translation = Translation::where([...Helper::langStatus()])->first();

    return array_merge(parent::share($request), [
      'appName' => config('app.name'),
      'appUrl' => config('app.url'),
      'auth' => [
        'user' => $request->user(),
        'has_password' => isset($request->user()?->password),
      ],
      'ziggy' => function () use ($request) {
        $ziggy = (new Ziggy)->toArray();
        unset($ziggy['routes']);
        return array_merge($ziggy, [
          'location' => $request->url(),
          'query' => count($request->query()) == 0 ? null : $request->query()
        ]);
      },
      'flash' => [
        'message' => fn() => $request->session()->get('message'),
        'success' => fn() => $request->session()->get('success'),
        'error' => fn() => $request->session()->get('error'),
      ],
      'location' => $request->fullUrl(),
      'locale' => $lang,
      'lang_translation' => $translation,
      'footer' => $this->getFooter($request),
      'default_meta' => $this->getDefaultMeta($request),
      'config' => $this->getConfig($request),
      'chat' => $this->getChat($request),
    ]);
  }

  public function getFooter($request)
  {
    $disabled = collect(['/api', '/admin']);
    $disabled = $disabled->filter(function ($endpoint) use ($request) {
      return Str::startsWith($request->getPathInfo(), $endpoint);
    });
    if ($disabled->isNotEmpty()) return null;

    $contacts = [
      ProfileType::Email => Helper::getCache("profile." . ProfileType::Email),
      ProfileType::Phone => Helper::getCache("profile." . ProfileType::Phone),
      ProfileType::Whatsapp => Helper::getCache("profile." . ProfileType::Whatsapp),
      ProfileType::Facebook => Helper::getCache("profile." . ProfileType::Facebook),
      ProfileType::Instagram => Helper::getCache("profile." . ProfileType::Instagram),
      ProfileType::Youtube => Helper::getCache("profile." . ProfileType::Youtube),
      ProfileType::OfficeAddress => Helper::getCache("profile." . ProfileType::OfficeAddress),
      ProfileType::WarehouseAddress => Helper::getCache("profile." . ProfileType::WarehouseAddress),
    ];

    $emptyContacts = collect($contacts)->filter(function ($contact) {
      return empty($contact);
    })->toArray();

    $profiles = [];

    if (count($emptyContacts) > 0) {
      $profiles = Profile::whereIn('type', array_keys($emptyContacts))
        ->where([...Helper::publishStatus(), ...Helper::langStatus()])
        ->select(['type', 'content'])
        ->get()
        ->groupBy('type');

      foreach ($profiles as $profile) {
        $profile = $profile->first();
        Helper::saveCache("profile.$profile->type", $profile);
      }
    }

    $contacts = collect($contacts)->map(function ($contact, $key) use ($profiles) {
      if (!empty($profiles[$key])) {
        $profile = $profiles[$key]->first();
        return $profile?->content;
      }
      if (!empty($contact)) {
        return $contact?->content;
      }
    });

    $data = [
      ...$contacts
    ];

    return $data;
  }
  public function getDefaultMeta($request)
  {
    $disabled = collect(['/api', '/admin']);
    $disabled = $disabled->filter(function ($endpoint) use ($request) {
      return Str::startsWith($request->getPathInfo(), $endpoint);
    });

    if ($disabled->isNotEmpty()) return null;

    $profileMeta = [
      ProfileType::CompanyName => Helper::getCache("profile." . ProfileType::CompanyName),
      ProfileType::About => Helper::getCache("profile." . ProfileType::About),
    ];

    $emptyProfileMeta = collect($profileMeta)->filter(function ($profile) {
      return empty($profile);
    })->toArray();

    $profiles = [];

    if (count($emptyProfileMeta) > 0) {
      $profiles = Profile::whereIn('type', array_keys($emptyProfileMeta))
        ->where([...Helper::publishStatus(), ...Helper::langStatus()])
        ->select(['type', 'content'])
        ->get()
        ->groupBy('type');

      foreach ($profiles as $profile) {
        $profile = $profile->first();
        Helper::saveCache("profile.$profile->type", $profile);
      }
    }

    $profileMeta = collect($profileMeta)->map(function ($profile, $key) use ($profiles) {
      if (!empty($profiles[$key])) {
        $profile = $profiles[$key]->first();
        return $profile?->content;
      }
      if (!empty($profile)) {
        return $profile?->content;
      }
    });

    $profileMeta = [
      ...$profileMeta,
      'seoKeywords' => null,
      'seoDescription' => null,
    ];

    if (!empty($profileMeta['about'])) {
      $profileMeta['about'] = Helper::htmlToString($profileMeta['about'], 160);
    }

    $defaultSeo = Helper::getCache('seo.' . SeoType::Default);
    if (!$defaultSeo) {
      $defaultSeo = Seo::where(['type' => SeoType::Default, ...Helper::publishStatus(), ...Helper::langStatus()])->first();
      Helper::saveCache("seo.$defaultSeo->type", $defaultSeo);
    }

    if ($defaultSeo) {
      $profileMeta['seoKeywords'] = $defaultSeo?->seo_keyword;
      $profileMeta['seoDescription'] = $defaultSeo?->seo_description ?: $profileMeta['about'];
    }

    $data = [
      ...$profileMeta
    ];

    return $data;
  }
  public function getConfig($request)
  {
    $disabled = collect(['/api', '/admin']);
    $disabled = $disabled->filter(function ($endpoint) use ($request) {
      return Str::startsWith($request->getPathInfo(), $endpoint);
    });
    if ($disabled->isNotEmpty()) return null;

    $toggles = [
      ConfigurationType::CS_CHAT
    ];

    $configs = [
      ConfigurationType::GOOGLE_VERIFICATION => Helper::getCache("config." . ConfigurationType::GOOGLE_VERIFICATION),
      ConfigurationType::GTM_ID => Helper::getCache("config." . ConfigurationType::GTM_ID),
      ConfigurationType::CTA_ACTION => Helper::getCache("config." . ConfigurationType::CTA_ACTION),
      ConfigurationType::CTA_FORM => Helper::getCache("config." . ConfigurationType::CTA_FORM),
      ConfigurationType::CS_CHAT => Helper::getCache("config." . ConfigurationType::CS_CHAT),
    ];

    $emptyConfigs = collect($configs)->filter(function ($config) {
      return empty($config);
    })->toArray();
    $configurations = [];

    if (count($emptyConfigs) > 0) {
      $configurations = Configuration::whereIn('type', array_keys($configs))
        ->where(['status' => OriginStatusType::Active])
        ->select(['type', 'value'])
        ->get()
        ->groupBy('type');

      foreach ($configurations as $configuration) {
        $configuration = $configuration->first();
        Helper::saveCache("config.$configuration->type", $configuration);
      }
    }

    $configs = collect($configs)->map(function ($config, $key) use ($configurations, $toggles) {
      $configuration = $config;
      if (!empty($configurations[$key])) {
        $configuration = $configurations[$key]->first();
      }

      if (empty($configuration)) return null;
      if (in_array($configuration->type, $toggles)) {
        return true;
      }
      return $configuration?->value;
    });


    $data = [
      ...$configs,
    ];

    return $data;
  }
  public function getChat($request)
  {
    $disabled = collect(['/api', '/admin']);
    $disabled = $disabled->filter(function ($endpoint) use ($request) {
      return Str::startsWith($request->getPathInfo(), $endpoint);
    });
    if ($disabled->isNotEmpty()) return null;

    $chat = null;

    if ($request->cookie('chat_id')) {
      $chat = Chat::where('id', base64_decode($request->cookie('chat_id')))
        ->with(['message' => ['imageAttachment', 'otherAttachment']])
        ->withCount('message')
        ->first();
    }

    return $chat;
  }
}
