<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Models\Category;
use App\Rules\UniqueSlug;
use App\Enums\CategoryType;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use Illuminate\Validation\Rule;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Http\Services\SoftDeleteService;

class CategoryController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $category = Category::filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "category" => collect($category),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Category/Index", $data);
  }

  protected function getFormData($request, $category = null)
  {
    $data = [
      "status" => PublishStatusType::getValues(),
      "types" => CategoryType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Category/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $category = Category::with('translatedGroup.category')->where("id", $id)->first();
    if (!$category) return Helper::redirectBack('error', 'Data tidak ditemukan');

    if (!$this->getAuthorize(request(), $category))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($category->translatedGroup->category as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $category->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $category = collect($category)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $formData = $this->getFormData($request, $category);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "category" => $category,
    ];

    return Inertia::render("Admin/Category/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        $data['slug'] = Helper::slugify($data['slug']);
        Category::create($data);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    if ($request->restore_data == '1') return SoftDeleteService::restore("categories", $id, $this->page);
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $category = Category::where("id", $id)->first();
      if (!$category) return Helper::redirectBack('error', 'Data tidak ditemukan');

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $category);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        if (empty($data['id'])) {
          continue;
        }
        $data['slug'] = Helper::slugify($data['slug']);

        $category = Category::firstWhere('id', $data['id']);
        $category->update($data);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $category = null)
  {
    if ($request->file('thumbnail')) {
      $directory = 'category';
      if (!Storage::disk('public')->exists($directory)) {
        Storage::disk('public')->makeDirectory($directory);
      }

      $fileExt = $request->file('thumbnail')->getClientOriginalExtension();
      $fileName = config('app.name') . "-" . strtolower($request->slug) . time() . ".$fileExt";
      $file = Storage::disk('public')->putFileAs($directory, $request->file('thumbnail'), $fileName);
      if (!$file) redirect()->back()->with('error', 'Gagal upload file');
      $validatedData['thumbnail'] = "/storage/$file";
    }

    $filteredData = [];
    if (empty($validatedData['type'])) {
      $validatedData['type'] = CategoryType::Product;
    }

    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Category
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $category = Category::with('translatedGroup.category')->whereId($id)->withTrashed()->first();
      if (!$category) return Helper::redirectBack('error', 'Data tidak ditemukan');

      if (!$this->getAuthorize(request(), $category))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      foreach ($category->translatedGroup->category as $translated) {
        if ($translated->deleted_at) {
          $translated->forceDelete();
        } else {
          $translated->delete();
        }
      }
      $category->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $category = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "category",
      "label" => "Kategori",
      "url" => "/admin/category",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $table = (new Category)->getTable();

    $data = [
      "validation" => [
        'id'  => "nullable",
        'lang'  => "required",
        'name'  => "required",
        "slug" => "required|unique:$table,slug",
        'type'  => "nullable",
        'thumbnail'  => "required",
        'seo_keyword'  => "nullable",
        'seo_description'  => "nullable",
        'description'  => "required",
        'status'  => "required",
      ],
      "default" => [
        "type" => CategoryType::Product,
        "lang" => LanguageType::Id,
        "status" => PublishStatusType::Draft,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "name" => [],
        "slug" => [],
        "description" => [],
        "seo_keyword" => [],
        "seo_description" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
            $data['validation']["{$column}_{$lang}"] = ['required', new UniqueSlug($table, $lang, [$request->input("id_$lang")])];
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
