<?php

namespace App\Http\Controllers\Admin;

use App\Models\Chat;
use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Enums\LanguageType;
use Illuminate\Http\Request;
use App\Enums\ChatStatusType;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class ChatController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $chat = Chat::filter($request)
      ->with(['message' => ['imageAttachment', 'otherAttachment']])
      ->withCount('message')
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "status" => ChatStatusType::getValues(),
      "page" => $this->page,
      "chat" => collect($chat),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Chat/Index", $data);
  }

  public function update($id, Request $request)
  {
    if ($request->filled('status') && $this->isAdmin) return $this->updateStatus($id, $request);
    return redirect()->back()->with('error', 'Aksi yang dipilih tidak ditemukan');
  }

  protected function updateStatus($id, $request)
  {
    DB::beginTransaction();
    try {
      $chat = Chat::where(['id' => $id])->first();
      $status = in_array($request->status, ChatStatusType::getValues()) ? $request->status : $chat->status;
      
      if (!empty($chat)) {
        $chat->update([
          'status' => $status
        ]);
      }

      DB::commit();
      return redirect()->back()->with("success", "Update Status Customer Support Berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Status Customer Support Gagal Di Update. Cobalah beberapa saat lagi");
    }
  }


  protected function getAuthorize($request, $chat = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "chat",
      "label" => "Customer Support",
      "url" => "/admin/chat",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $data = [
      "validation" => [],
      "default" => [
        "status" => PublishStatusType::Draft,
      ],
      "langs" => [],
      "lang_columns" => []
    ];

    return $data;
  }
}
