<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Models\Brand;
use App\Utils\Helper;
use App\Enums\UserType;
use App\Models\Product;
use App\Enums\ImageType;
use App\Models\Category;
use App\Rules\UniqueSlug;
use App\Enums\ProductType;
use App\Enums\CategoryType;
use App\Enums\LanguageType;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use App\Enums\TranslatedType;
use App\Models\TranslatedGroup;
use Illuminate\Validation\Rule;
use App\Enums\PublishStatusType;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Http\Services\SoftDeleteService;

class ProductController extends Controller
{
  public $adminRole = [UserType::Admin];
  public $isAdmin = false;
  public $page = null;

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = auth()->user();
      $this->page = $this->getPage($request);
      $this->isAdmin = $user && in_array($user->role, $this->adminRole);
      return $next($request);
    });
  }

  public function index(Request $request)
  {
    $lang = Helper::getLangActive();

    $product = Product::with('brand', 'category', 'productImage')
      ->filter($request)
      ->when($request->filled("status"), function ($query) use ($request) {
        $query->where("status", $request->status);
      })
      ->where('lang', $lang)
      ->paginate(20);

    $data = [
      "title" => "{$this->page["label"]}",
      "langs" => LanguageType::getValues(),
      "lang" => $lang,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "product" => collect($product),
      "isAdmin" => $this->isAdmin,
    ];

    return Inertia::render("Admin/Product/Index", $data);
  }

  protected function getFormData($request, $product = null)
  {
    $brands = Brand::published()->select(["id", "id as value", "name as label"])->get();
    $allCategories = Category::select(["id", "lang", "id as value", "name as label"])
      ->where(["type" => CategoryType::Product, "status" => PublishStatusType::Publish])
      ->get();

    $categories = [];
    $validation = $this->validation($request);
    foreach ($validation['langs'] as $lang) {
      $categories[$lang] = $allCategories->where('lang', $lang)->values();
    }

    $data = [
      "brands" => $brands,
      "categories" => $categories,
      "status" => PublishStatusType::getValues(),
      "page" => $this->page,
      "isAdmin" => $this->isAdmin,
    ];
    return $data;
  }

  public function create()
  {
    $formData = $this->getFormData(request());
    $data = [
      ...$formData,
      "title" => "Tambah {$this->page["label"]}",
    ];

    return Inertia::render("Admin/Product/Create", $data);
  }

  public function edit($id, Request $request)
  {
    $product = Product::with('translatedGroup.product', 'productImage')->where("id", $id)->first();
    if (!$product) return Helper::redirectBack('error', 'Data tidak ditemukan');

    if (!$this->getAuthorize(request(), $product))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $langColumns = array_keys($this->validation($request)['lang_columns']);

    foreach ($product->translatedGroup->product as $translated) {
      $translateds = $translated->only($langColumns);
      foreach ($translateds as $key => $value) {
        $product->{"{$key}_{$translated->lang}"} = $value;
      }
    }

    $product = collect($product)->filter(function ($value, $key) use ($langColumns) {
      return !in_array($key, $langColumns);
    })->toArray();

    $product['images'] = $product['product_image'];

    $formData = $this->getFormData($request, $product);
    $data = [
      ...$formData,
      "title" => "Edit {$this->page["label"]}",
      "product" => $product,
    ];

    return Inertia::render("Admin/Product/Edit", $data);
  }

  public function store(Request $request)
  {
    if (!$this->getAuthorize($request))
      return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

    $validation = $this->validation($request);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $validationResult = $this->processValidateData($validatedData, $validation, $request);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        unset($data['images']);
        $data['slug'] = Helper::slugify($data['slug']);
        $product = Product::create($data);
        $this->saveProductImage($request, $product);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Tambah {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Tambah {$this->page["label"]} gagal");
    }
  }

  public function update($id, Request $request)
  {
    if (!$this->getAuthorize($request)) return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");
    if ($request->restore_data == '1') return SoftDeleteService::restore("categories", $id, $this->page);
    $validation = $this->validation($request, $id);
    $validatedData = $request->validate($validation["validation"]);

    DB::beginTransaction();
    try {
      $product = Product::where("id", $id)->first();
      if (!$product) return Helper::redirectBack('error', 'Data tidak ditemukan');

      $validationResult = $this->processValidateData($validatedData, $validation, $request, $product);
      if ($validationResult instanceof \Illuminate\Http\RedirectResponse) return $validationResult;

      [$validatedData] = $validationResult;

      foreach ($validatedData as $data) {
        unset($data['images']);
        if (empty($data['id'])) {
          continue;
        }
        $data['slug'] = Helper::slugify($data['slug']);

        $product = Product::firstWhere('id', $data['id']);
        $product->update($data);
        $this->saveProductImage($request, $product);
      }

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Update {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Kesalahan Server. Update {$this->page["label"]} gagal");
    }
  }

  protected function processValidateData($validatedData, $validation, $request, $product = null)
  {
    $validatedData = $this->saveBrand($validatedData);

    $filteredData = [];
    if (empty($validatedData['translated_group_id'])) {
      $translatedGroup = TranslatedGroup::create([
        'type' => TranslatedType::Product
      ]);
      $validatedData['translated_group_id'] = $translatedGroup->id;
    }

    if (!empty($validation['langs'] || !empty($validation['lang_columns']))) {
      $langColumns = collect($validation['lang_columns'])->flatten()->toArray();
      $baseColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return !in_array($key, $langColumns);
      })->toArray();
      $langColumns = collect($validatedData)->filter(function ($value, $key) use ($langColumns) {
        return in_array($key, $langColumns);
      })->toArray();

      foreach ($validation['langs'] as $lang) {
        $validatedColumns = $baseColumns;
        $validatedColumns['lang'] = $lang;

        foreach ($validation['lang_columns'] as $column => $values) {
          $validatedColumns[$column] = $langColumns["{$column}_{$lang}"];
        }
        $filteredData[] = $validatedColumns;
      }
    }

    if (!empty($filteredData)) {
      $validatedData = $filteredData;
    }

    return [$validatedData];
  }

  public function saveBrand($validatedData)
  {
    if (gettype($validatedData['brand_id']) != 'array') return $validatedData;
    $name = $validatedData['brand_id']['value'];
    $brand = Brand::create([
      'name' => $name,
      'slug' => str($name)->slug()->value
    ]);
    $validatedData['brand_id'] = $brand->id;
    return $validatedData;
  }

  public function saveProductImage($request, $product)
  {
    $images = $request->file('images');

    $latestImage = collect($request->images)->pluck('id')->whereNotNull()->toArray();
    $productImage = ProductImage::where('product_id', $product->id)->whereNotIn('id', $latestImage)->get();

    if ($productImage->isNotEmpty()) {
      foreach ($productImage as $image) {
        if (Storage::disk('public')->exists(substr($image->value, 9))) {
          Storage::disk('public')->delete(substr($image->value, 9));
        }
        $image->delete();
      }
    }

    if (!$images) return;

    $directory = 'product';
    if (!Storage::disk('public')->exists($directory)) {
      Storage::disk('public')->makeDirectory($directory);
    }

    if (!empty($images)) {
      foreach ($images as $image) {
        $fileName = config('app.name') . "-" . str($image->getClientOriginalName())->slug()->value()  . "-" . time(). "-" . rand(111,999) . "." . $image->getClientOriginalExtension();
        $file = Storage::disk('public')->putFileAs($directory, $image, $fileName);
        if (!$file) redirect()->back()->with('error', 'Gagal upload file');

        if (!isset($image->id)) {
          ProductImage::create([
            'user_id' => auth()->id(),
            'product_id' => $product->id,
            'type' => ImageType::File,
            'label' => $image->getClientOriginalName(),
            'value' => "/storage/$file"
          ]);
        }
      }
    }
  }

  public function destroy($id)
  {
    DB::beginTransaction();
    try {
      $product = Product::with('translatedGroup.product')->whereId($id)->first();
      if (!$product) return Helper::redirectBack('error', 'Data tidak ditemukan');

      if (!$this->getAuthorize(request(), $product))
        return redirect()->back()->with("error", "Anda tidak dapat mengakses fitur ini");

      foreach ($product->translatedGroup->product as $translated) {
        $translated->delete();
      }

      $product->translatedGroup->delete();

      DB::commit();
      return redirect(\App\Utils\Helper::getRefurl(request()) ?? $this->page['url'])->with("success", "Hapus {$this->page["label"]} berhasil");
    } catch (\Throwable $th) {
      DB::rollBack();
      return redirect()->back()->with("error", "Hapus {$this->page["label"]} gagal, {$this->page["label"]} ini masih digunakan fitur lain");
    }
  }

  protected function getAuthorize($request, $product = null)
  {
    if ($this->isAdmin) return true;
    return true;
  }
  protected function getPage($request, $id = null)
  {
    $fields = \App\Utils\Helper::getFormFields($this->validation($request));

    $page = [
      "name" => "product",
      "label" => "Produk",
      "url" => "/admin/product",
      "data" => null,
      "fields" => $fields,
    ];

    return $page;
  }
  protected function validation($request, $id = null)
  {
    $table = (new Product)->getTable();

    $data = [
      "validation" => [
        'brand_id'  => "required",
        'category_id'  => "required",
        'id'  => "nullable",
        'lang'  => "required",
        'name'  => "required",
        "slug" => "required|unique:$table,slug",
        'seo_keyword'  => "nullable",
        'seo_description'  => "nullable",
        'description'  => "required",
        'status'  => "required",
        'images'  => "required",
      ],
      "default" => [
        "lang" => LanguageType::Id,
        "status" => PublishStatusType::Draft,
      ],
      "langs" => LanguageType::getValues(),
      "lang_columns" => [
        "id" => [],
        "category_id" => [],
        "name" => [],
        "slug" => [],
        "description" => [],
        "seo_keyword" => [],
        "seo_description" => [],
      ]
    ];

    if (!empty($data['langs'] || !empty($data['lang_columns']))) {
      foreach ($data['lang_columns'] as $column => $value) {
        foreach ($data['langs'] as $lang) {
          $data['validation']["{$column}_{$lang}"] = $data['validation'][$column];
          if ($column == 'slug' && $request->filled("id_$lang")) {
            $data['validation']["{$column}_{$lang}"] = ['required', new UniqueSlug($table, $lang, [$request->input("id_$lang")])];
          }
          $data['lang_columns'][$column][] = "{$column}_{$lang}";
        }
        unset($data['validation'][$column]);
      }
    }

    return $data;
  }
}
