<?php

namespace App\Http\Controllers\User;

use Carbon\Carbon;
use App\Models\Blog;
use Inertia\Inertia;
use App\Utils\Helper;
use App\Enums\SeoType;
use App\Models\Profile;
use App\Models\BlogImage;
use App\Enums\ProfileType;
use App\Enums\LanguageType;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use App\Enums\PublishStatusType;
use App\Http\Services\SeoService;
use App\Http\Controllers\Controller;
use App\Http\Services\TranslationService;

class BlogController extends Controller
{
  public $companyName = '';
  public $baseFilter = [];
  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $this->baseFilter = [...Helper::publishStatus(), ...Helper::langStatus()];

      $this->companyName = config('app.name');
      $companyNameProfile = Profile::where(['type' => ProfileType::CompanyName, ...$this->baseFilter])->select(['type', 'content'])->first();
      if (!empty($companyNameProfile?->content)) {
        $this->companyName = $companyNameProfile?->content;
      }
      return $next($request);
    });
  }
  public function index()
  {
    $featuredCategory = BlogCategory::where([...$this->baseFilter])->select('name')->where('is_featured', 1)->first();
    $featuredBlogs = Blog::where([...$this->baseFilter])
      ->whereRelation('category', 'is_featured', '=', 1)
      ->orderByDesc('published_at')
      ->get();

    $categories = $this->blogCategories(false);
    $blogCount = Blog::published()->where('published_at', '<', now())->count();

    $data = [
      ...SeoService::get(SeoType::BlogIndex),
      'featuredCategory' => $featuredCategory,
      'featuredBlogs' => $featuredBlogs,
      'categories' => $categories,
      'blogCount' => $blogCount,
    ];

    // dd($data);

    return Inertia::render('User/Blog/Index', $data);
  }

  public function show(string $slug, Request $request)
  {
    $blog = Blog::with('user', 'category', 'blogImage')
      ->where('slug', $slug)
      ->when($request->mode != 'preview', function ($query) {
        return $query->published()->where('published_at', '<', now());
      })
      ->first();

    $langStatus = Helper::langStatus();  
    if ($langStatus['lang'] != $blog->lang) {
      $blog = Blog::where("translated_group_id", $blog->translated_group_id)
        ->where($langStatus)
        ->when($request->mode != 'preview', function ($query) {
          return $query->published()->where('published_at', '<', now());
        })
        ->first();
        
      if ($blog) {
        return redirect("/blog/{$blog->slug}");
      }
    }

    if (!$blog) return redirect('/blog')->with('error', 'Blog tidak ditemukan');
    if ($request->mode == 'preview' && Helper::authorize($blog->user_id) != true) {
      return redirect()->back()->with('error', 'Anda tidak memiliki akses untuk melihat artikel ini');
    }

    $blog->short_description ??= Helper::htmlToString($blog->description, 160);
    // $blog->published_at = Carbon::createFromDate($blog->published_at)->diffForHumans();

    $relatedBlogs = Blog::with('user')->where('category_id', $blog->category_id)->published()->where('published_at', '<', now())->where('id', '!=', $blog->id)->orderByDesc('published_at')->take(5)->get();

    $image = $blog->thumbnail;

    if (!$image && $blog->blogImage->count() > 0) {
      $image = $blog->blogImage->first()->value;
      $blog->thumbnail = $image;
    }

    if ($image) {
      $image = str($image)->startsWith('/storage') ? config('app.url') . $image : $image;
      $image = str($image)->startsWith('/image') ? config('app.url') . $image : $image;
    }

    $keywords = $blog->keyword ?? '';

    $meta = [
      'title' => $blog->title,
      'description' => $blog->short_description,
      'image' => $image,
      'keywords' => $keywords,
    ];

    $blogUrl = config('app.url') . "/blog/{$blog->slug}";
    $data = [
      'title' => $blog->title,
      'blog' => $blog,
      'relatedBlogs' => $relatedBlogs,
      'meta' => $meta,
      'blogUrl' => $blogUrl,
      'schema' => $this->getBlogSchema($blog, $meta),
    ];

    $minutes = now()->diffInMinutes(now()->addYear());
    $langCookie = cookie('language', $blog?->lang ?: LanguageType::Id, $minutes, '/', null, false, false);

    $response = Inertia::render('User/Blog/Show', $data)->toResponse($request);
    return $response->withCookie($langCookie);
  }

  public function categoryShow(string $slug, Request $request)
  {
    $category = BlogCategory::where('slug', $slug)->first();
    if (!$category) return redirect('/blog')->with('error', 'Kategori tidak ditemukan');
    $blogs = Blog::with('user')->where('category_id', $category?->id)->published()->where('published_at', '<', now())->orderByDesc('published_at')->paginate(20);
    $blogs = collect($blogs);

    $blogs['data'] = collect($blogs['data'])->map(function ($blog) {
      $blog['short_description'] ??= Helper::htmlToString($blog['description'], 160);
      // $blog['published_at'] = Carbon::createFromDate($blog['published_at'])->diffForHumans();

      $image = $blog['thumbnail'] ?? $blog['banner_image'];

      if (!$image) {
        $blogImage = BlogImage::where('blog_id', $blog['id'])->where('type', 'slider')->first();
        if ($blogImage) {
          $image = $blogImage->value;
        }
        $blog['thumbnail'] = $image;
      }

      return $blog;
    });

    $categoryImage = str($category->thumbnail)->startsWith('/storage') ? config('app.url') . $category->thumbnail : $category->thumbnail;
    $categoryImage = str($category->thumbnail)->startsWith('/image') ? config('app.url') . $category->thumbnail : $category->thumbnail;

    $meta = [
      'title' => $category->name,
      'description' => $category->seo_description,
      'keywords' => $category->seo_keyword,
      'image' => $categoryImage,
    ];

    $data = [
      'title' => $category->name,
      'category' => $category,
      'blogs' => $blogs,
      'meta' => $meta,
      'schema' => $this->getCategorySchema($category, $blogs, $meta),
    ];

    $minutes = now()->diffInMinutes(now()->addYear());
    $langCookie = cookie('language', $category?->lang ?: LanguageType::Id, $minutes, '/', null, false, false);

    $response = Inertia::render('User/Blog/Category', $data)->toResponse($request);
    return $response->withCookie($langCookie);
  }

  public function blogCategories($isFeatured = true)
  {
    $categories = BlogCategory::where([...$this->baseFilter])
      ->when(!$isFeatured, function ($query) {
        $query->where('is_featured', 0);
      })->with([
        'blog' => function ($query) {
          $query->with('user')->published()->where('published_at', '<', now())->orderByDesc('published_at')->take(5);
        }
      ])->get();

    $categories = $categories->map(function ($category) {
      $category->blog = $category->blog->map(function ($blog) {
        $blog->short_description ??= Helper::htmlToString($blog->description, 160);
        // $blog->published_at = Carbon::createFromDate($blog->published_at)->diffForHumans();

        $image = $blog->thumbnail;

        if (!$image && $blog->blogImage->count() > 0) {
          $image = $blog->blogImage->first()->value;
          $blog->thumbnail = $image;
        }
        return $blog;
      });

      return $category;
    });

    return $categories;
  }

  public function getCategorySchema($category, $blogs, $meta)
  {
    $image = str($category->thumbnail)->startsWith('/storage') ? config('app.url') . $category->thumbnail : $category->thumbnail;
    $image = str($category->thumbnail)->startsWith('/image') ? config('app.url') . $category->thumbnail : $category->thumbnail;

    $companyName = 'PT. Gayamakmur Techno Nusantara';
    $companyNameProfile = Helper::getCache('profile.' . ProfileType::CompanyName);
    if (!empty($companyNameProfile)) {
      $companyName = $companyNameProfile->content ?? $companyName;
    }
    $appUrl = rtrim(config('app.url'), '/');
    $posts = collect($blogs['data'])->toArray();
    $categoryUrl = "{$appUrl}/blog/category/{$category->slug}";

    $data = [
      "@context" => "https://schema.org",
      "@type" => "CollectionPage",
      "@id" => $categoryUrl,
      "url" => $categoryUrl,
      "name" => $category->name,
      "description" => $meta['description'] ?? null,
      "keywords" => $meta['keywords'] ?? null,
      "image" => $image,
      "inLanguage" => "id",
      "isPartOf" => [
        "@type" => "WebSite",
        "@id" => "{$appUrl}#website",
        "url" => $appUrl,
        "name" => $companyName,
      ],
      "breadcrumb" => [
        "@type" => "BreadcrumbList",
        "itemListElement" => [
          ["@type" => "ListItem", "position" => 1, "name" => "Home", "item" => $appUrl],
          ["@type" => "ListItem", "position" => 2, "name" => "Blog", "item" => "{$appUrl}/blog"],
          ["@type" => "ListItem", "position" => 3, "name" => $category->name, "item" => $categoryUrl]
        ]
      ],
      "mainEntity" => [
        "@type" => "ItemList",
        "itemListOrder" => "https://schema.org/ItemListOrderDescending",
        "numberOfItems" => count($posts),
        "itemListElement" => array_map(function ($post, $i) use ($appUrl) {
          return [
            "@type" => "ListItem",
            "position" => $i + 1,
            "item" => [
              "@type" => "BlogPosting",
              "@id" => "{$appUrl}/blog/{$post['slug']}",
              "url" => "{$appUrl}/blog/{$post['slug']}",
              "headline" => $post['title'],
              "description" => $post['short_description'],
              "keywords" => $post['keyword'],
              "image" => "{$appUrl}{$post['thumbnail']}",
              "datePublished" => date(DATE_ISO8601, strtotime($post['published_at'])),
              "dateModified" => date(DATE_ISO8601, strtotime($post['updated_at'])),
              "author" => [
                "@type" => "Organization",
                "@id" => "{$appUrl}#organization"
              ]
            ]
          ];
        }, $posts, array_keys($posts))
      ]
    ];

    $data = Helper::emptyArrayFilter($data);

    $schema = '<script type="application/ld+json">';
    $schema .= json_encode($data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    $schema .= '</script>';
    return $schema;
  }

  public function getBlogSchema($blog, $meta)
  {
    $author = $blog->user?->name ?? '';
    $publishedAt = Carbon::createFromDate($blog->published_at);
    $thumbnail = $blog->thumbnail;
    $thumbnail = str($thumbnail)->startsWith('/storage') ? config('app.url') . $thumbnail : $thumbnail;
    $thumbnail = str($thumbnail)->startsWith('/image') ? config('app.url') . $thumbnail : $thumbnail;

    $blogCategory = $blog->category->name;

    $companyName = 'PT. Gayamakmur Techno Nusantara';
    $companyNameProfile = Helper::getCache('profile.' . ProfileType::CompanyName);
    if (!empty($companyNameProfile)) {
      $companyName = $companyNameProfile->content ?? $companyName;
    }

    $appUrl = rtrim(config('app.url'), '/');
    $blogUrl = "$appUrl/blog/$blog->slug";
    $imageUrl = $appUrl . $blog->thumbnail;

    $content = Helper::htmlToString($blog->description);

    $data = [
      "@context" => "https://schema.org",
      "@type" => "NewsArticle",
      "@id" => $blogUrl,
      "mainEntityOfPage" => [
        "@type" => "WebPage",
        "@id" => $blogUrl
      ],
      "headline" => $blog->title,
      "keywords" => $meta['keywords'],
      "description" => $meta['description'],
      "articleSection" => $blogCategory,
      "datePublished" => date(DATE_ISO8601, strtotime($blog->published_at)),
      "dateModified" => date(DATE_ISO8601, strtotime($blog->updated_at)),
      "image" => [
        "@type" => "ImageObject",
        "url" => $imageUrl,
        "width" => 1200,
        "height" => 675
      ],
      "thumbnailUrl" => $imageUrl,
      "wordCount" => str_word_count($content),
      "articleBody" => $content,
      "copyrightYear" => date('Y', strtotime($blog->published_at)),
      "inLanguage" => "id",
      "author" => [
        "@type" => "Person",
        "name" => $author
      ],
      "publisher" => [
        "@type" => "Organization",
        "@id" => "$appUrl#organization",
        "name" => $companyName,
        "url" => $appUrl,
        "logo" => [
          "@type" => "ImageObject",
          "url" => $appUrl . "/image/icon.png",
          "width" => 60,
          "height" => 60
        ]
      ],
      "copyrightHolder" => [
        "@type" => "Organization",
        "name" => $companyName,
        "url" => $appUrl,
        "logo" => [
          "@type" => "ImageObject",
          "url" => $appUrl . "/image/icon.png",
          "width" => 60,
          "height" => 60
        ]
      ],
      "isPartOf" => [
        "@type" => "WebSite",
        "@id" => $appUrl,
        "url" => $appUrl,
        "name" => $companyName,
        "inLanguage" => "id"
      ]
    ];
    $data = Helper::emptyArrayFilter($data);

    $schema = '<script type="application/ld+json">';
    $schema .= json_encode($data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    $schema .= '</script>';
    return $schema;
  }
}
