<?php

namespace App\Http\Services;

use Carbon\Carbon;
use App\Models\Seo;
use App\Utils\Helper;
use App\Enums\SeoType;
use App\Enums\ProfileType;
use App\Models\Profile;

class SchemaService
{
  public static function getSchema($metaData)
  {
    $appUrl = rtrim(config('app.url'), '/');
    $currentUrl = url()->current();
    $imageUrl = $appUrl . '/image/icon.png';
    $logoUrl = $appUrl . '/image/icon.png';

    $companyName = 'PT. Gayamakmur Techno Nusantara';
    $companyNameProfile = Helper::getCache('profile.' . ProfileType::CompanyName);
    if (!empty($companyNameProfile)) {
      $companyName = $companyNameProfile->content ?? $companyName;
    }
    $address = Helper::getCache('profile.' . ProfileType::OfficeAddress)?->content;
    if ($address) {
      $address = Helper::htmlToString($address);
    }

    $type = $metaData['type'] ?? 'Organization';

    $data = [
      "@type" => $type,
    ];

    $defaultSeo = Helper::getCache('seo.' . SeoType::Default);

    if ($type === 'Organization') {
      $data = self::getOrganization($defaultSeo, $companyName, $appUrl, $logoUrl, $address);
    }

    if ($type === 'Product') {
      $data += [
        "name" => $metaData['title'] ?? $companyName,
        "description" => $metaData['description'] ?? null,
        "keywords" => $metaData['keywords'] ?? null,
        "image" => [
          $metaData['image'] ?? $imageUrl,
        ],
        "brand" => [
          "@type" => "Brand",
          "name" => $metaData['brand'] ?? $companyName,
        ],
        "offers" => [
          "@type" => "Offer",
          "url" => $currentUrl,
          "availability" => "https://schema.org/InStock",
          "itemCondition" => "https://schema.org/NewCondition",
        ],
      ];

      if (!empty($metaData['price'])) {
        $productSchema["offers"]["price"] = $metaData['price'];
        $productSchema["offers"]["priceCurrency"] = $metaData['currency'] ?? "IDR";
      }

      if (!empty($metaData['ratingValue']) && !empty($metaData['reviewCount'])) {
        $data['aggregateRating'] = [
          "@type" => "AggregateRating",
          "ratingValue" => $metaData['ratingValue'],
          "reviewCount" => $metaData['reviewCount'],
        ];
      }
    }

    if ($type === 'Service') {
      $data += [
        "@id" => "$currentUrl#service",
        "name" => $metaData['title'] ?? $companyName,
        "description" => $metaData['description'] ?? null,
        "keywords" => $metaData['keywords'] ?? null,
        "image" => [
          $metaData['image'] ?? $imageUrl,
        ],
        "brand" => [
          "@type" => "Brand",
          "name" => $metaData['brand'] ?? $companyName,
        ],
        "provider" => self::getOrganization($defaultSeo, $companyName, $appUrl, $logoUrl, $address),
      ];

      if (!empty($metaData['ratingValue']) && !empty($metaData['reviewCount'])) {
        $data['aggregateRating'] = [
          "@type" => "AggregateRating",
          "ratingValue" => $metaData['ratingValue'],
          "reviewCount" => $metaData['reviewCount'],
        ];
      }
    }

    if (in_array($type, ['Article', 'NewsArticle', 'BlogPosting'])) {
      $data += [
        "mainEntityOfPage" => [
          "@type" => "WebPage",
          "@id" => $currentUrl,
        ],
        "headline" => $metaData['title'] ?? config('app.name'),
        "description" => $metaData['description'] ?? null,
        "keywords" => $metaData['keywords'] ?? null,
        "datePublished" => $metaData['datePublished'] ?? null,
        "dateModified" => $metaData['dateModified'] ?? null,
        "image" => [
          "@type" => "ImageObject",
          "url" => $imageUrl,
        ],
        "author" => [
          "@type" => "Organization",
          "name" => $companyName,
          "url" => $appUrl,
        ],
        "publisher" => [
          "@type" => "Organization",
          "name" => $companyName,
          "logo" => [
            "@type" => "ImageObject",
            "url" => $logoUrl,
          ],
        ],
        "copyrightHolder" => [
          "@type" => "Organization",
          "name" => $companyName,
          "url" => $appUrl,
          "logo" => [
            "@type" => "ImageObject",
            "url" => $appUrl . "/image/icon.png",
            "width" => 60,
            "height" => 60
          ]
        ],
        "isPartOf" => [
          "@type" => "WebSite",
          "@id" => $appUrl,
          "url" => $appUrl,
          "name" => $companyName,
          "inLanguage" => "id"
        ],
      ];
    }


    $graph = [];
    $graph[] = $data;

    if ($type == 'Service') {
      $graph[] =     [
        "@type" => "WebPage",
        "@id" => "$currentUrl#webpage",
        "url" => "$currentUrl",
        "mainEntity" => [
          "@id" => "$currentUrl#service"
        ]
      ];
    }

    if ($type != 'Organization') {
      $graph[] = self::getOrganization($defaultSeo, $companyName, $appUrl, $logoUrl, $address);
    }

    $graph[] = [
      "@type" => "WebSite",
      "url" => $appUrl,
      "name" => $companyName,
      "publisher" => [
        "@type" => "Organization",
        "@id" => "$appUrl#organization",
      ]
    ];

    $graph[] = [
      "@type" => "LocalBusiness",
      "name" => $companyName,
      "image" => $logoUrl,
      "url" => $appUrl,
      "telephone" => Helper::getCache('profile.' . ProfileType::Phone)?->content,
      "hasMap" => Helper::getCache('profile.' . ProfileType::AddressMap)?->content,
      "description" => $defaultSeo?->seo_description,
      'email' => Helper::getCache('profile.' . ProfileType::Email)?->content ?: null,
      "address" => [
        "@type" => "PostalAddress",
        "streetAddress" => $address,
        "addressRegion" => Helper::getCache('profile.' . ProfileType::Location)?->content,
        "addressCountry" => "ID",
      ],
      "parentOrganization" => [
        "@type" => "Organization",
        "@id" => "$appUrl#organization",
      ]
    ];

    $graph = Helper::emptyArrayFilter($graph);
    $schemaData = [
      "@context" => "https://schema.org",
      "@graph" => $graph
    ];

    $schema = '<script type="application/ld+json">';
    $schema .= json_encode($schemaData, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    $schema .= '</script>';

    return $schema;
  }

  public static function getOrganization($defaultSeo, $companyName, $appUrl, $logoUrl, $address)
  {
    $whatsapp = Helper::getCache('profile.' . ProfileType::Whatsapp)?->content;
    if ($whatsapp) {
      $whatsapp = Helper::whatsappNumber($whatsapp);
      $whatsapp = "+$whatsapp";
    }

    $location = Helper::getCache('profile.' . ProfileType::Location)?->content;
    if (!$location) {
      $locationProfile = Profile::select(['type', 'content'])->where(['type' => ProfileType::Location, ...Helper::publishStatus(), ...Helper::langStatus()])->first();
      if ($locationProfile) {
        Helper::saveCache("profile.$locationProfile->type", $locationProfile);
        $location = $locationProfile->content;
      }
    }

    $foundedYear = Helper::getCache('profile.' . ProfileType::FoundedYear)?->content;
    if (!$foundedYear) {
      $foundedYearProfile = Profile::select(['type', 'content'])->where(['type' => ProfileType::FoundedYear, ...Helper::publishStatus(), ...Helper::langStatus()])->first();
      if ($foundedYearProfile) {
        Helper::saveCache("profile.$foundedYearProfile->type", $foundedYearProfile);
        $foundedYear = $foundedYearProfile->content;
      }
    }

    return [
      "@type" => "Organization",
      "@id" => "$appUrl#organization",
      "name" => $companyName,
      "url" => $appUrl,
      "logo" => [
        "@type" => "ImageObject",
        "url" => $logoUrl,
      ],
      "description" => $defaultSeo?->seo_description ?? null,
      "keywords" => $defaultSeo?->seo_keyword ?? null,
      "foundingDate" => $foundedYear,
      'email' => Helper::getCache('profile.' . ProfileType::Email)?->content ?: null,
      "address" => [
        "@type" => "PostalAddress",
        "streetAddress" => $address,
        "addressRegion" => $location,
        "addressCountry" => "ID",
      ],
      "sameAs" => array_filter([
        Helper::getCache('profile.' . ProfileType::Facebook)?->content,
        Helper::getCache('profile.' . ProfileType::Instagram)?->content,
        Helper::getCache('profile.' . ProfileType::Youtube)?->content,
      ]),
      "contactPoint" => [
        "@type" => "ContactPoint",
        "telephone" => $whatsapp,
        'email' => Helper::getCache('profile.' . ProfileType::Email)?->content ?: null,
        "contactType" => "Customer Service",
        "areaServed" => "ID",
        "availableLanguage" => ["Indonesian", "English"],
      ]
    ];
  }
}
