import TextArea from "@/Components/Form/TextArea"
import { router, useForm, usePage } from "@inertiajs/react"
import { useRef, useState } from "react"
import { dateHumanize, deleteCookie } from "@/utlis/format"
import { useEffect } from "react"
import ImageGallery from "react-image-gallery"
import "react-image-gallery/styles/css/image-gallery.css"
import { IoIosClose } from "react-icons/io"
import useLang from "@/utlis/useLang"

function UserChatModal({ ModalButton = null }) {
  let { props } = usePage()
  const [show, setShow] = useState(null)
  const handleShow = (data) => {
    setShow(data)
  }
  if (!props?.config?.CS_CHAT) return <></>

  return (
    <>
      <div
        role="button"
        tabIndex={0}
        onClick={() => handleShow(props?.chat ?? "new")}
        className={ModalButton ? "" : "fixed bottom-24 lg:bottom-14 2xl:bottom-0 w-max right-1 lg:right-5 3xl:w-full 3xl:right-auto 3xl:mb-14 z-[102] 3xl:z-[100]"}
      >
        {ModalButton ? <ModalButton /> : <DefaultModalButton />}
      </div>

      {show ? (
        <>
          <ChatModal data={props?.chat ?? "new"} handleClick={() => handleShow(null)} />
        </>
      ) : null}
    </>
  )
}

const DefaultModalButton = () => {
  return (
    <div className="mx-auto max-w-screen-3xl flex justify-end gap-2 pr-5 ">
      <button aria-label="chat" className="bg-green-500 text-white px-3 py-3 rounded-full shadow-lg hover:bg-green-600 transition-colors">
        <i className="fas fa-comment fa-2x"></i>
      </button>
    </div>
  )
}

const ChatModal = (state) => {
  const { auth } = usePage().props
  const { data: chat, handleClick } = state
  const bottomRef = useRef(null)

  const [attachment, setAttachment] = useState(false)
  const [isFullScreen, setIsFullScreen] = useState(false)
  const { data, setData, post, errors, processing } = useForm({
    chat_id: chat?.id ?? "new",
    message: "",
    attachment: ""
  })

  function handleChange(e) {
    const { name, value } = e.target

    setData((state) => ({
      ...state,
      [name]: value
    }))
  }

  useEffect(() => {
    bottomRef.current?.scrollIntoView({ behavior: "smooth" })
    return () => {}
  }, [])

  useEffect(() => {
    let intervalId
    if (!!chat?.id && ["open", "new"].includes(chat?.status)) {
      setData((state) => ({
        ...state,
        chat_id: chat?.id
      }))

      intervalId = setInterval(() => {
        fetch(`/chat/${chat.id}/status`)
          .then((res) => res.json())
          .then((res) => {
            let latestMessage = res?.data?.chat?.message?.slice(-1)[0]
            let currentLatestMessage = chat?.message?.slice(-1)[0]

            if (res?.data?.is_new || latestMessage?.id != currentLatestMessage?.id) {
              router.reload({
                only: ["chat"],
                async: false,
                onSuccess: () => {
                  bottomRef.current?.scrollIntoView({ behavior: "smooth" })
                }
              })
            }
          })
          .catch(function (error) {
            console.log(error)
          })
      }, 5000)
    }

    return () => (intervalId ? clearInterval(intervalId) : null)
  }, [chat?.id, chat?.status])

  const handleSubmit = (e) => {
    e.preventDefault()
    data.attachment = ""

    post(`/chat`, {
      preserveScroll: true,
      onSuccess: () => {
        setData((state) => ({
          ...state,
          ["message"]: "",
          ["attachment"]: ""
        }))
        setAttachment(false)
        router.reload({
          only: ["chat"],
          async: false,
          onSuccess: () => bottomRef.current?.scrollIntoView({ behavior: "smooth" })
        })
      }
    })
  }

  const handleNewChat = () => {
    deleteCookie("chat_id")
    router.reload({ only: ["chat"], async: false })
  }

  const inputProps = { handleChange, data, errors }

  return (
    <div className="fixed lg:right-5 3xl:right-auto bottom-10 md:bottom-20 lg:bottom-28 w-full z-[200]  flex flex-row-reverse    ">
      <div className="w-full h-full mx-auto flex flex-row-reverse  max-w-screen-3xl">
        <div className="transition-all  w-full duration-300 md:w-96 2xl:w-[600px]  min-h-96    xl:max-w-screen-3xl   " id="chat">
          <div className="w-full max-w-screen-md mx-auto md:max-h-[80vh] h-auto p-2 md:p-4 text-black">
            <form className="bg-gradient-to-t from-[#c8e8f6] to-white  rounded-xl shadow-sm flex flex-col h-full">
              {/* Header */}
              <div className="flex items-center justify-between bg-[#0399DB] rounded-t-xl p-3">
                <p className="font-semibold text-lg text-white">Customer Service</p>
                <button type="button" onClick={handleClick} className="hover:scale-110 transition-transform duration-300">
                  <IoIosClose size={32} color="white" />
                </button>
              </div>

              {/* Chat Body */}
              <div
                className={`flex-1 overflow-auto p-3      ${
                  attachment
                    ? "max-h-[50vh] md:max-h-10"
                    : "min-h-[40vh] md:min-h-[30vh] max-h-[50vh] md:max-h-10 2xl:min-h-[380px] 2xl:max-h-[380px]"
                }`}
              >
                {chat !== "new" &&
                  chat?.message?.length > 0 &&
                  chat.message.map((chat, index) => {
                    const images = chat.image_attachment?.map((attachment) => ({
                      original: attachment?.value,
                      thumbnail: attachment?.value,
                      originalClass: `w-full p-2 object-contain ${isFullScreen ? "" : "max-h-[300px]"}`
                    }))

                    const galleryOptions = {
                      showThumbnails: false,
                      showBullets: true,
                      showFullscreenButton: true,
                      showPlayButton: false,
                      onScreenChange: (fullscreen) => setIsFullScreen(fullscreen)
                    }

                    return (
                      <div
                        key={index}
                        className={`p-2 my-2 rounded-lg break-words ${
                          chat?.user_id == auth?.user?.id
                            ? "text-right bg-[#0399DB] text-white"
                            : "text-left bg-white border text-[#0399DB] border-[#0399DB]"
                        }`}
                      >
                        {images?.length > 0 && <ImageGallery items={images} {...galleryOptions} />}

                        {chat.other_attachment?.length > 0 && (
                          <div className="mt-2">
                            {chat.other_attachment
                              .sort((a, b) => b.type.localeCompare(a.type))
                              .map((attachment, i) => (
                                <a
                                  key={i}
                                  href={attachment.value}
                                  target="_blank"
                                  rel="noopener noreferrer"
                                  className="text-sm inline-block mt-1 text-white hover:underline"
                                >
                                  <i className="fas fa-download mr-2"></i>
                                  {attachment?.type === "file" ? attachment?.label : <span>Buka Link</span>}
                                </a>
                              ))}
                          </div>
                        )}

                        <p>{chat.message}</p>
                        <p className="text-xs opacity-80 mt-1">{dateHumanize(chat.created_at)}</p>
                        {index + 1 >= state?.data?.message?.length && <div ref={bottomRef} />}
                      </div>
                    )
                  })}
              </div>

              {/* Message Box or Closed Chat Message */}
              {chat?.status === "close" ? (
                <p className="text-center p-3">{useLang("chat.form.close_label")}</p>
              ) : (
                <div className="w-full px-3 pb-2">
                  <TextArea name="message" placeHolder={useLang("chat.form.placeholder")} isLabel={false} {...inputProps} />
                </div>
              )}

              {/* Action Button */}
              <div className="p-3 flex flex-row-reverse">
                {chat?.status !== "close" ? (
                  <button
                    onClick={handleSubmit}
                    disabled={processing}
                    className="bg-[#0399DB] text-white py-1 px-4 rounded-md hover:bg-white hover:text-[#0399DB] border border-[#0399DB] transition-all duration-300"
                  >
                    <span className={`${processing ? "loading" : ""}`}>{useLang("chat.form.send")}</span>
                  </button>
                ) : (
                  <button
                    type="button"
                    onClick={handleNewChat}
                    className="bg-[#0399DB] text-white py-1 px-4 rounded-md hover:bg-white hover:text-[#0399DB] border border-[#0399DB] transition-all duration-300"
                  >
                    {useLang("chat.form.new_chat")}
                  </button>
                )}
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  )
}

export default UserChatModal
