import { htmlParse } from "@/utlis/format";
import useLang from "@/utlis/useLang";
import { useEffect, useState } from "react";

// --- FUNGSI-FUNGSI BANTU UNTUK ANALISIS ---

// Heuristik sederhana untuk menghitung suku kata (syllable)
const countSyllables = (word) => {
  word = word.toLowerCase();
  if (word.length <= 3) return 1;
  word = word.replace(/(?:[^laeiouy]es|ed|[^laeiouy]e)$/, '');
  word = word.replace(/^y/, '');
  const matches = word.match(/[aeiouy]{1,2}/g);
  return matches ? matches.length : 0;
};

// Daftar kata transisi untuk EN dan ID
const transitionWords = {
  id: new Set(['selain itu', 'namun', 'akan tetapi', 'oleh karena itu', 'dengan demikian', 'akibatnya', 'selanjutnya', 'maka', 'jadi', 'misalnya', 'sebagai contoh', 'pertama', 'kedua', 'akhirnya']),
  en: new Set(['in addition', 'however', 'but', 'therefore', 'consequently', 'as a result', 'next', 'so', 'for example', 'for instance', 'firstly', 'secondly', 'finally', 'above all', 'in conclusion']),
};

// Pola regex sederhana untuk mendeteksi kalimat pasif
const passiveVoicePatterns = {
  id: /\bdi([a-z]+)\b/g, // Pola "di-verb"
  en: /\b(is|are|was|were|be|been|being)\s+([a-zA-Z]+ed|spoken|written|given|made|taken)\b/g,
};

// --- KOMPONEN UTAMA ---

function ReadabilityEvaluator({ data, lang }) {
  const [analysis, setAnalysis] = useState(null);

  useEffect(() => {
    if (data) {
      // Mengambil konten mentah (tanpa HTML)
      const content = htmlParse(data?.[`description_${lang}`]?.toLowerCase() || '');
      if (!content) return;

      // 1. Segmentasi Teks
      const sentences = content.match(/[^.!?]+[.!?]+/g) || [];
      const words = content.split(/\s+/).filter(Boolean);
      const paragraphs = content.split(/\n\n+/).filter(Boolean);

      // 2. Analisis Metrik
      const sentenceCount = sentences.length;
      const wordCount = words.length;
      const syllableCount = words.reduce((acc, word) => acc + countSyllables(word), 0);
      
      // Analisis Panjang Kalimat
      const longSentences = sentences.filter(s => s.split(/\s+/).length > 20).length;
      const sentenceLengthScore = (longSentences / sentenceCount) * 100;

      // Analisis Panjang Paragraf
      const longParagraphs = paragraphs.filter(p => (p.match(/[^.!?]+[.!?]+/g) || []).length > 5).length;
      const paragraphLengthScore = (longParagraphs / paragraphs.length) * 100;

      // Analisis Kata Transisi
      const transitionWordSentences = sentences.filter(s =>
        [...transitionWords[lang]].some(tw => s.includes(tw))
      ).length;
      const transitionWordsPercentage = (transitionWordSentences / sentenceCount) * 100;
      
      // Analisis Kalimat Pasif
      const passiveSentences = sentences.filter(s =>
        s.match(passiveVoicePatterns[lang])
      ).length;
      const passiveVoicePercentage = (passiveSentences / sentenceCount) * 100;

      // 3. Flesch Reading Ease Score
      // Rumus: 206.835 - 1.015 * (total_words / total_sentences) - 84.6 * (total_syllables / total_words)
      let fleschScore = 0;
      if (sentenceCount > 0 && wordCount > 0) {
        fleschScore = 206.835 - 1.015 * (wordCount / sentenceCount) - 84.6 * (syllableCount / wordCount);
      }
      
      setAnalysis({
        sentenceLength: { percentage: sentenceLengthScore.toFixed(1), isGood: sentenceLengthScore <= 25 },
        paragraphLength: { percentage: paragraphLengthScore.toFixed(1), isGood: paragraphLengthScore <= 25 },
        transitionWords: { percentage: transitionWordsPercentage.toFixed(1), isGood: transitionWordsPercentage >= 30 },
        passiveVoice: { percentage: passiveVoicePercentage.toFixed(1), isGood: passiveVoicePercentage <= 10 },
        fleschScore: { score: fleschScore.toFixed(1), label: getFleschLabel(fleschScore) },
      });
    }
  }, [data, lang]);

  const getFleschLabel = (score) => {
    if (score >= 90) return { text: "Sangat Mudah", color: "text-green-600" };
    if (score >= 70) return { text: "Mudah Dibaca", color: "text-green-600" };
    if (score >= 60) return { text: "Standar", color: "text-yellow-600" };
    if (score >= 50) return { text: "Agak Sulit", color: "text-orange-600" };
    if (score >= 30) return { text: "Sulit Dibaca", color: "text-red-600" };
    return { text: "Sangat Sulit", color: "text-red-600" };
  };

  if (!analysis) return null;

  return (
    <div className="p-1 bg-white shadow rounded space-y-1">
      <h2 className="text-xl font-semibold">📊 Analisis Keterbacaan ({useLang(`lang.${lang}`)})</h2>
      
      <div className="space-y-1 mt-2">
        <ul className="list-disc list-inside">
          <li className={analysis.fleschScore.label.color}>
            Skor Flesch Reading: {analysis.fleschScore.score} ({analysis.fleschScore.label.text})
          </li>
          <li className={analysis.passiveVoice.isGood ? "text-green-600" : "text-red-600"}>
            Kalimat Pasif: {analysis.passiveVoice.percentage}% {analysis.passiveVoice.isGood ? '✅' : '❌'} (Target: di bawah 10%)
          </li>
          <li className={analysis.transitionWords.isGood ? "text-green-600" : "text-red-600"}>
            Kata Transisi: {analysis.transitionWords.percentage}% {analysis.transitionWords.isGood ? '✅' : '❌'} (Target: di atas 30%)
          </li>
          <li className={analysis.sentenceLength.isGood ? "text-green-600" : "text-red-600"}>
            Panjang Kalimat: {analysis.sentenceLength.percentage}% kalimat terlalu panjang {analysis.sentenceLength.isGood ? '✅' : '❌'}
          </li>
          <li className={analysis.paragraphLength.isGood ? "text-green-600" : "text-red-600"}>
            Panjang Paragraf: {analysis.paragraphLength.percentage}% paragraf terlalu panjang {analysis.paragraphLength.isGood ? '✅' : '❌'}
          </li>
        </ul>
      </div>
    </div>
  );
}

export default ReadabilityEvaluator