import { disableButtonAtom } from '@/atoms';
import FormBuilder from '@/Components/FormBuilder';
import { currency, useSearchParams } from '@/utlis/format';
import useFormBuilder from '@/utlis/useFormBuilder';
import useLang from '@/utlis/useLang';
import { router, usePage } from '@inertiajs/react';
import { useSetAtom } from 'jotai';
import { useEffect, useState } from 'react';

function MainForm({ isEdit = false, submitLabel = null }) {
  const {props} = usePage()
  let onSubmit = null

  if (isEdit) {
    onSubmit = (e, form)=> {
      const {data, put} = form
      data.editDraft = '1'
      return put(`${props?.page?.url}/${props?.budget?.id}`);
    }
  }

  if (props?.formType == 'edit-submit') {
    onSubmit = (e, form)=> {
      const {data, put} = form
      data.submitEdit = '1'
      return put(`${props?.page?.url}/${props?.budget?.id}`);
    }
  }

  const formConfig = { isEdit, getProperties, onSubmit }
  const builderProps = useFormBuilder(formConfig)

  return (
    <FormBuilder {...builderProps} submitLabel={submitLabel} />
  )
}

const getProperties = (state) => {
  const { setData, props, data } = state
  const { params } = useSearchParams(props.location)  
  let formType = props?.formType ?? 'create'
  
  const typeOptions = props.type.map((data) => {
    return {
      label: useLang(`budget.type.${data}`, props),
      value: data,
    }
  })

  const handleSelectReload = (e, name, only) => {
    let newState = {}
    newState[name] = e?.value ?? ''   
    
    if (!newState[name]) {
      newState[name] = e ?? ''   
    } 
    if (name == 'artist_id') {
      newState['budget_id'] = null
    }

    setData((state)=>({  ...state,  ...newState}))

    params.set(name, newState[name])
    router.visit(`?${params.toString()}`, {
      method: 'get',
      only: [...only, 'location'],
      preserveState: true,
      async: false
    })
  }

  const properties = [
    [
      {
        form: 'select',
        isHidden: !props.isAdmin || ['credit'].includes(data?.type) || ['edit-submit'].includes(formType),
        props: {
          isReactSelect: true,
          name: 'user_id',
          label: 'Pilih Admin Divisi',
          options: props.users,
          handleChange: (e) => setData('user_id', e?.value),
        }
      },
      {
        form: 'select',
        isHidden: ['edit-submit'].includes(formType),
        props: {
          isReactSelect: true,
          name: 'type',
          label: 'Jenis Anggaran',
          options: typeOptions,
          handleChange: (e) => handleSelectReload(e, 'type', ['budgets','categories','page']),
        }
      },
      {
        form: 'select',
        isHidden: ['edit-submit'].includes(formType),
        props: {
          isReactSelect: true,
          isVirtualized:true,
          name: 'artist_id',
          label: 'Artis',
          options: props.artists,
          handleChange: (e) => handleSelectReload(e, 'artist_id', ['budgets','categories','selected_artist']),
        }
      },
      {
        form: 'DatePicker',
        props: {
          name: 'year',
          label: 'Tahun Anggaran',
          type:'year',
          className:'h-[2.4rem] rounded-sm border-gray-200',
          handleChange: (e) => handleSelectReload(e, 'year', ['budgets','selected_artist']),
        }
      },
      {
        form: 'select',
        isHidden: !['','new'].includes(data.type) || ['edit-submit'].includes(formType),
        props: {
          isReactSelect: true,
          name: 'category_id',
          label: 'Kategori',
          options: props.categories,
          handleChange: (e) => setData('category_id', e?.value),
        }
      },
      {
        form: 'select',
        isHidden: !['revision','credit'].includes(data.type),
        props: {
          isReactSelect: true,
          name: 'budget_id',
          label: 'Kode Anggaran',
          options: props.budgets,
          isDisabled: !props.selected_artist,
          noOptionsMessage: `Belum ada anggaran yang ditambahkan atau belum disetujui ${props.selected_artist ? `untuk ${props.selected_artist?.NAMA}` : ''}`,
          handleChange: (e) => handleSelectReload(e, 'budget_id', ['selected_budget']),
        }
      },
    ],
    [
      {
        form: 'input',
        isHidden: ['credit'].includes(data?.type),
        props: {
          name: 'debit_amount',
          label: 'Jumlah',
          type: 'tel',
        }
      },
      {
        form: 'input',
        isHidden: !['credit'].includes(data?.type),
        props: {
          name: 'credit_amount',
          label: 'Jumlah',
          type: 'tel',
        }
      },
      {
        form: 'input',
        props: {
          name: 'date',
          label: 'Tanggal',
          type: 'date',
        }
      },
    ],
    [
      {
        form: 'textarea',
        props: {
          name: 'description',
          label: 'Keterangan',
        },
      },
      {
        form: 'textarea',
        isHidden: formType != 'edit-submit',
        props: {
          name: 'note',
          label: 'Catatan Pengeditan',
        }
      },
    ],
    [
      {
        custom: (props) => <RemainingBudgetCard {...props} />
      },
    ],
  ]

  return properties
}

const RemainingBudgetCard = (props) => {
  const { artists, selected_budget } = usePage().props
  const {data} = props.form

  if (!data?.artist_id) return <></>
  if (!data?.budget_id) return <></>

  const artist = artists?.find((artist)=>artist?.value == data.artist_id)  

  const setDisableButton = useSetAtom(disableButtonAtom)
  const [isLimit, setIsLimit] = useState(false)


  useEffect(() => {
    if (['credit'].includes(data.type)) {
      setIsLimit(parseInt(data?.credit_amount ?? 0) > parseInt(selected_budget?.budget_category?.remaining_amount ?? 0))
    } 

    return () => {}
  }, [data.credit_amount, data.type, selected_budget])

  useEffect(() => {
    setDisableButton(isLimit)
    return () => {}
  }, [isLimit])

  return (
    <div className="w-full">
      <p className='text-sm'>Catatan: </p>

      {['revision'].includes(data.type) ? (
        <>
          <p>Sejumlah <span className='font-semibold'>{currency(data.debit_amount ?? 0, 'id-ID', selected_budget?.category?.currency_code ?? 'IDR')}</span> akan ditambahkan untuk revisi atau penambahan Anggaran <span className='font-semibold'>{selected_budget?.code} - {selected_budget?.description}</span> pada kategori <span className='font-semibold'>{selected_budget?.category?.name}</span></p>
        </>
      ) : null}

      {['credit'].includes(data.type) ? (
        <>
          <p>Sejumlah <span className='font-semibold'>{currency(data.credit_amount ?? 0, 'id-ID', selected_budget?.category?.currency_code ?? 'IDR')}</span> akan mengurangi sisa Anggaran <span className='font-semibold'>{selected_budget?.code} - {selected_budget?.description}</span> pada kategori <span className='font-semibold'>{selected_budget?.category?.name}</span> yang tersedia sejumlah <span className='font-semibold'>{currency(selected_budget?.budget_category?.remaining_amount ?? 0, 'id-ID', selected_budget?.category?.currency_code ?? 'IDR')}</span></p>
          {isLimit ? (
            <p className='text-red-600'>Jumlah {currency(data.credit_amount ?? 0, 'id-ID', selected_budget?.category?.currency_code ?? 'IDR')} melebihi jumlah pada anggaran yang dipilih yaitu maksimal {currency(selected_budget?.budget_category?.remaining_amount ?? 0, 'id-ID', selected_budget?.category?.currency_code ?? 'IDR')}</p>
          ) : null}
        </>
      ) : null}
    </div>
  )
}

export default MainForm
